#ifndef __CAN_H_
#define __CAN_H_

#include <fcntl.h>
#include <sys/ioctl.h>

#ifdef  __cplusplus
extern "C" {
#endif  //__cplusplus 



typedef struct {
    /** flags, indicating or controlling special message properties */
    int             flags;
    int             cob;   /**< CAN object number, used in Full CAN  */
    unsigned   long id;     /**< CAN message ID, 4 bytes  */
    struct timeval  timestamp;   /**< time stamp for received messages */
    short      int  length;   /**< number of bytes in the CAN message */
    unsigned   char data[8]; /**< data, 0...8 bytes */
} canmsg_t;

struct Command_par {
  int cmd;      /**< special driver command */
  int target;      /**< special configuration target */
  unsigned long val1;    /**< 1. parameter for the target */
  unsigned long val2;    /**< 2. parameter for the target */
  int error;       /**< return value */
  unsigned long retval;  /**< return value */
};

/**
IOCTL Command request parameter structure */
typedef struct Command_par Command_par_t ; /**< Command parameter struct */
/**
IOCTL CConfiguration request parameter structure */
typedef struct Command_par  Config_par_t ; /**< Configuration parameter struct */

#define CAN_MSG_LENGTH 8                /**< maximum length of a CAN frame */
#define MSG_ACTIVE      (0)             /**< Controller Error Active */
#define MSG_BASE        (0)             /**< Base Frame Format */
#define MSG_RTR         (1<<0)          /**< RTR Message */
#define MSG_OVR         (1<<1)          /**< CAN controller Msg overflow error */
#define MSG_EXT         (1<<2)          /**< extended message format */
#define MSG_SELF        (1<<3)          /**< message received from own tx */
#define MSG_PASSIVE     (1<<4)          /**< controller in error passive */
#define MSG_BUSOFF      (1<<5)          /**< controller Bus Off  */
#define MSG_WARNING     (1<<6)          /**< CAN Warning Level reached */
#define MSG_BOVR        (1<<7)          /**< receive/transmit buffer overflow */


//###########################################################################
//# AGC Linux API Defines                                                   #
//###########################################################################
#define MAX_DEV  1

#define TRUE 1
#define FALSE 0

#define MAX_PORT_NUM PORT_NUM_L

#define MEMBASE    0x10
#define IOBASE     0x14

typedef unsigned char   u8;
typedef int   i32;
typedef char  i8;
typedef short   i16;
typedef unsigned short u16;
typedef unsigned int u32;

#define ARBIOC_MAGIC  'k' /* Use 'k' as magic number */
#define ARBIOC_IOR_BYTE _IOR(ARBIOC_MAGIC,  0, unsigned int)
#define ARBIOC_IOR_WORD _IOR(ARBIOC_MAGIC,  1, unsigned int)
#define ARBIOC_IOR_DWRD _IOR(ARBIOC_MAGIC,  2, unsigned int)
#define ARBIOC_IOW_BYTE _IOW(ARBIOC_MAGIC,  3, unsigned int)
#define ARBIOC_IOW_WORD _IOW(ARBIOC_MAGIC,  4, unsigned int)
#define ARBIOC_IOW_DWRD _IOW(ARBIOC_MAGIC,  5, unsigned int)
#define ARBIOC_MMR_BYTE _IOR(ARBIOC_MAGIC,  6, unsigned int)
#define ARBIOC_MMR_WORD _IOR(ARBIOC_MAGIC,  7, unsigned int)
#define ARBIOC_MMR_DWRD _IOR(ARBIOC_MAGIC,  8, unsigned int)
#define ARBIOC_MMW_BYTE _IOW(ARBIOC_MAGIC,  9, unsigned int)
#define ARBIOC_MMW_WORD _IOW(ARBIOC_MAGIC,  10, unsigned int)
#define ARBIOC_MMW_DWRD _IOW(ARBIOC_MAGIC,  11, unsigned int)

#define ARBIOC_CFR_BYTE  _IOR(ARBIOC_MAGIC, 12, unsigned int)
#define ARBIOC_CFR_WORD  _IOR(ARBIOC_MAGIC, 13, unsigned int)
#define ARBIOC_CFR_DWRD  _IOR(ARBIOC_MAGIC, 14, unsigned int)
#define ARBIOC_CFW_BYTE  _IOW(ARBIOC_MAGIC, 15, unsigned int)
#define ARBIOC_CFW_WORD  _IOW(ARBIOC_MAGIC, 16, unsigned int)
#define ARBIOC_CFW_DWRD  _IOW(ARBIOC_MAGIC, 17, unsigned int)
#define ARBIOC_GET_CONT  _IOR(ARBIOC_MAGIC, 18, unsigned int)
#define ARBIOC_GET_BUFF  _IOR(ARBIOC_MAGIC, 19, unsigned int)

#define ARBIOC_CAN_COMMAND _IOR(ARBIOC_MAGIC, 20, unsigned int)
#define ARBIOC_CAN_CONFIG  _IOR(ARBIOC_MAGIC, 21, unsigned int)
#define ARBIOC_CAN_SEND    _IOR(ARBIOC_MAGIC, 22, unsigned int)
#define ARBIOC_CAN_STATUS  _IOR(ARBIOC_MAGIC, 23, unsigned int)

struct IO_ACCESS {
  unsigned long offset;
  union {
    unsigned char  byte;
    unsigned short word;
    unsigned int   dword;
    } value;
};

typedef struct _event {
  u16 Type;     //Event Type
  u16 Channel;   //Related Channel/Timer Number
  u16 Value;    //Related Value
} event;

#define INPUT_NO   12
#define OUTPUT_NO  12
#define TIMER_NO   16
#define INPUT       0x00
#define COUNTER_TO  0x01
#define COUNTER_OF  0x02
#define PG_FIN      0x03
#define TIMER_FIN   0x04
#define COUNTER_REACH 0x05

//###########################################################################
//# AGC Linux API Functions                                                 #
//###########################################################################

unsigned int sendCanMessages( canmsg_t* Buffer, u8 Count);
unsigned int getCanMessages( canmsg_t* Buffer, u8 Count);
unsigned int configCan( int Baud );

#ifdef  __cplusplus
}
#endif  //__cplusplus 

#endif  // __CAN_H_
