#include <sys/io.h>
#include <stdio.h>
#include <stdlib.h>
#include <curses.h>
#include <semaphore.h>
#include <pthread.h>
#include <unistd.h>
#include <time.h>

#include "sio_acce.h"


/* For User Interface */
int running = 0;

sem_t plotting;
time_t start;
int baseLine = 0;

WINDOW *subWinLeft  = NULL;
WINDOW *subWinR1 = NULL;
WINDOW *subWinR2 = NULL;
WINDOW *subWinR3 = NULL;

static void *scanReg(void *);
static void *inputCmd(void *);

static void printLeftText(void);

static void showWinText(WINDOW *win, int lines, int cols, char *str);
static void showBits(WINDOW *win, int lines, int cols, unsigned char num);
static void showShortBits(WINDOW *win, int lines, int cols, unsigned char num);

/* End */


unsigned int delayTime = 100000;

int main(int argc, char *argv[])
{

  int subLinesLeft = 0;
  int subLinesRight = 0;
  int subCols = 0;

  pthread_t scanningRegThd;
  pthread_t inputingCmdThd;

  if( argc == 2 )
    delayTime = atoi(argv[1]);

  initscr();

  
  /*
   * Start colors
   */
  if( !has_colors() ) {
    fprintf(stderr, "No color support");
    exit(1);
  }
  
  if( start_color() != OK ) {
    fprintf( stderr, "Start color fail");
    exit(1);
  }

  subLinesLeft  = LINES - 4;
  subLinesRight = (LINES - 10) / 3;
  subCols       = (COLS - 7) / 2;


  /* Initialize and set up the Windows */
  subWinLeft = newwin( subLinesLeft, subCols, 2, 2);
  scrollok( subWinLeft, 1);

  subWinR1 = newwin( subLinesRight, subCols, 2, 2+subCols+3);
  scrollok( subWinR1, 1);

  subWinR2 = newwin( subLinesRight, subCols, 2+subLinesRight+3, 2+subCols+3);
  scrollok( subWinR2, 1);

  subWinR3 = newwin( subLinesRight, subCols, 2+(subLinesRight+3)*2, 2+subCols+3);
  scrollok( subWinR3, 1);

  /* Plot Frame */
  init_pair(3, COLOR_BLACK, COLOR_YELLOW );  
  init_pair(70, COLOR_WHITE,  COLOR_BLACK);

  wattrset( stdscr, COLOR_PAIR(3) );

  move(0, 0);
  hline(' ', COLS); // Top herizon
  move(LINES-1, 0);
  hline(' ', COLS); // Bottom herizon

  move(1, 0); // left 
  vline(' ', LINES-2);
  move(1, 2+subCols+1); // middle
  vline(' ', LINES-2);
  move(1, COLS-1); // right
  vline(' ', LINES-2);

  move(2+subLinesRight+1, 2+subCols+1);
  hline(' ', subCols+4); 
  move(2+subLinesRight+3+subLinesRight+1, 2+subCols+1);
  hline(' ', subCols+4); 
  refresh();

  wattrset( stdscr, COLOR_PAIR(70) );
  /* End */

  sem_init(&plotting, 0, 1);

  /* Print out the main items at the left window. */
  printLeftText();


  running = 1;
  pthread_create(&scanningRegThd, NULL, scanReg, NULL );
  pthread_create(&inputingCmdThd, NULL, inputCmd, NULL );

  pthread_join( inputingCmdThd, NULL );
  pthread_join( scanningRegThd, NULL );

  sem_destroy( &plotting );

  delwin(subWinR3);
  delwin(subWinR2);
  delwin(subWinR1);
  delwin(subWinLeft);

  endwin();

  exit(0);
}

void *inputCmd(void *ptr)
{
  char num = 0;
  int count = 0;

  while( running )
  {

    wscanw(subWinLeft, "%d", &num);
    wrefresh(subWinLeft);

    if( num != 0 ) {

      /* Deal with wrong input */
      if( num != 1 && num != 2 && num != 3 ) {
        showWinText(subWinLeft, baseLine+2, 0, "Wrong number!");
        sleep(1);
        werase(subWinLeft);
        printLeftText();
      
        continue;
      }

      switch( num ) {
        case 1:  // Configure GPIO output
          /* ---- switch(num), case 1 ---- */
          while( num != 0 ) {
            static unsigned char value = 1;

            showWinText(subWinLeft, baseLine+0, 0, "\n\n\n\n\n\n\n\n\n\n\n\n");
            showWinText(subWinLeft, baseLine+0,  0, "Now set GPO to ");
            wprintw(subWinLeft, "%d. Choose the pin:\n", value); 
            showWinText(subWinLeft, baseLine+2, 0, "(0) GPO0\n");
            showWinText(subWinLeft, baseLine+3, 0, "(1) GPO1\n");
            showWinText(subWinLeft, baseLine+4, 0, "(2) GPO2\n");
            showWinText(subWinLeft, baseLine+5, 0, "(3) GPO3\n");
            showWinText(subWinLeft, baseLine+6, 0, "(4) Reset GPO to 0\n");

            showWinText(subWinLeft, baseLine+8, 0, "(9) Exit\n\n");

            /* Get the input here */
            wscanw(subWinLeft, "%d", &num);
            wrefresh(subWinLeft);
        
            if( num == 9 ) {
              werase(subWinLeft);
              printLeftText();

              break; /* Quitting the 'while( num !=0 )... */

            } else {
              iopl(3);
              switch(num)
              {
                case 0:
                      getOutChLevel( GPO0, &value);
                      value = (value+1)%2;
                      setOutChLevel( GPO0, value);
                      num = 5;
                      break;

                case 1:
                      getOutChLevel( GPO1, &value);
                      value = value >> 1;
                      value = (value+1)%2;
                      setOutChLevel( GPO1, value);
                      break;

                case 2:
                      getOutChLevel( GPO2, &value);
                      value = value >> 2;
                      value = (value+1)%2;
                      setOutChLevel( GPO2, value);
                      break;

                case 3:
                      getOutChLevel( GPO3, &value);
                      value = value >> 3;
                      value = (value+1)%2;
                      setOutChLevel( GPO3, value);
                      break;

                case 4:
                      setOutChLevel( GPOALL, 0);
                      break;

                default:
                      num = 5;
                      continue;
              } // End of 'switch(num)'
            } // End of 'if( num == 9 )...else...'
          } // End of 'while( num != 0 )...'
          break;
  
        case 2:    // Configure Watchdog
  
          /* ---- switch(num) case 2 ---- */
          showWinText(subWinLeft, baseLine+0, 0, "\n\n\n\n\n\n\n\n\n\n");
          showWinText(subWinLeft, baseLine+0, 0, "Input the countdown in sceonds: ");

          wscanw(subWinLeft, "%d", &count);
          wrefresh(subWinLeft);
  
          if( count == 0 ) {
            sem_wait(&plotting);
            wmove(subWinLeft, 8, 14);
            wprintw(subWinLeft, "\n");
            wrefresh(subWinLeft);
            sem_post(&plotting);

            start = (time_t) 0;
            setWtdTimer( (unsigned char) count);

          } else {
            sem_wait(&plotting);
            wmove(subWinLeft, 8, 14);
            wprintw(subWinLeft, "The counter starts from %d", count);
            wrefresh(subWinLeft);
            sem_post(&plotting);
            setWtdTimer( (unsigned char) count);
            time( &start );
          }
          break;

        default:
          break;

      } // End of switch()

      showWinText(subWinLeft, baseLine+0, 0, "Please input: "); 
    } else {
      running = 0;
    } // End of 'if( num != 0 )'
  }

  pthread_exit(NULL);
}

void *scanReg(void *ptr)
{
  unsigned char value = 0;
  time_t  now = (time_t) 0;

  while( running )
  {
    sem_wait(&plotting);

    /* GPI 0~3 */
    werase( subWinR1 );
    getInChLevel( GPIALL, &value);

    wmove(subWinR1, 2, 1);
    wprintw(subWinR1, "GPIO IN %2x", value);
    showShortBits(subWinR1, 3, 1, value);
    wrefresh(subWinR1);
    value = 0;

    /* GPO 0~3 */
    werase( subWinR2 );
    getOutChLevel( GPOALL, &value);
    wmove(subWinR2, 2, 1);
    wprintw(subWinR2, "GPIO OUT %2x", value);
    showShortBits(subWinR2, 3, 1, value);
    wrefresh(subWinR2);
    value = 0;

    /* Watchdog */
    value = getWtdTimer();
    werase( subWinR3 );
    wmove(subWinR3, 2, 1);
    wprintw(subWinR3, "Watchdog Counter %3d   ", value );
    if( start != 0 ) {
      time( &now );
      wprintw(subWinR3, "( %3.2f ", difftime( now, start) );
      wprintw(subWinR3, "seconds elapses... )");
    } else {
      wprintw(subWinR3, "\n");
    }

    showBits(subWinR3, 3, 1, value);
    wrefresh(subWinR3);

    sem_post(&plotting);

    usleep(delayTime);
  }

  pthread_exit(NULL);

}

static void showWinText(WINDOW *win, int lines, int cols, char *str)
{
  sem_wait(&plotting); // <----------
  wmove(win, lines, cols);
  wprintw(win, "\n");
  wmove(win, lines, cols);
  wprintw(win, "%s", str);
  wrefresh(win);
  sem_post(&plotting); // ---------->

  return;
}

static void showBits(WINDOW *win, int lines, int cols, unsigned char num)
{
  int i;
  int bits[8] = {0};

  for(i=0; i<8; i++) {
    if( (num & (0x1<<i) ) == (0x1<<i) )
      bits[i] = 1;
  }


  wmove( win, lines, cols);

  for(i=0; i<4; i++) {
    wprintw(win, " %d", bits[7-i]);
  }

  wprintw(win, "  ");

  for(i=4; i<8; i++) {
    wprintw(win, " %d", bits[7-i]);
  }

  return;
}


static void showShortBits(WINDOW *win, int lines, int cols, unsigned char num)
{
  int i;
  int bits[8] = {0};

  for(i=0; i<8; i++) {
    if( (num & (0x1<<i) ) == (0x1<<i) )
      bits[i] = 1;
  }

  wmove( win, lines, cols);

  for(i=4; i<8; i++) {
    wprintw(win, " %d", bits[7-i]);
  }

  return;
}


void printLeftText(void)
{
  sem_wait(&plotting);

  wmove(subWinLeft, 0, 0);
  wprintw(subWinLeft, "(0) Exit\n");
  wprintw(subWinLeft, "(1) Set GPIO Output.\n");
  wprintw(subWinLeft, "(2) Set Watchdog Timer.\n");

  wprintw(subWinLeft, "\nInformation: ");

  baseLine = 8;

  wmove(subWinLeft, baseLine+0, 0);
  wprintw(subWinLeft, "Please input: ");
  
  wrefresh(subWinLeft);

  sem_post(&plotting);

  return;
}
