/*

    bttv-i2c.c  --  all the i2c code is here

    bttv - Bt848 frame grabber driver

    Copyright (C) 1996,97,98 Ralph  Metzler (rjkm@thp.uni-koeln.de)
			   & Marcus Metzler (mocm@thp.uni-koeln.de)
    (c) 1999-2003 Gerd Knorr <kraxel@bytesex.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/delay.h>

#include "bttvp.h"
#include <media/v4l2-common.h>
#include <linux/jiffies.h>
#include <asm/io.h>

// ST_1120
#include "techwell.h"
static struct i2c_algo_bit_data	TW_i2c_algo_bit_template;
static struct i2c_adapter 		TW_i2c_adap_sw_template;
static struct i2c_client 			TW_i2c_client_template;
static struct i2c_adapter 		TW_i2c_adap_hw_template;


static int attach_inform(struct i2c_client *client);

static int i2c_debug;
static int i2c_hw;
static int i2c_scan;
module_param(i2c_debug, int, 0644);
module_param(i2c_hw,    int, 0444);
module_param(i2c_scan,  int, 0444);
MODULE_PARM_DESC(i2c_scan,"scan i2c bus at insmod time");

/* ----------------------------------------------------------------------- */
/* I2C functions - bitbanging adapter (software i2c)                       */

static void TW6801_bit_setscl(void *data, int state)
{
	struct bttv *btv = (struct bttv*)data;

	if (state)
		btv->i2c_state |= 0x01 << 6; 
	else
		btv->i2c_state &= ~(0x01 << 6);

	// ST_1121 for I2C Control 
	// btwrite(btv->i2c_state, BT848_I2C);
	// btread(BT848_I2C);
	btwrite(btv->i2c_state, TW6801_SBUSC );
	btread(TW6801_SBUSC);
	

}

static void TW6801_bit_setsda(void *data, int state)
{
	struct bttv *btv = (struct bttv*)data;

	if (state)
		btv->i2c_state |= 0x01 << 7;
	else
		btv->i2c_state &= ~(0x01 << 7);

	// ST_1121
	// btwrite(btv->i2c_state, BT848_I2C);
	// btread(BT848_I2C);
	btwrite( btv->i2c_state, TW6801_SBUSC ) ;
	btread( TW6801_SBUSC ) ;
	
}


static int TW6801_bit_getscl(void *data)
{
	struct bttv *btv = (struct bttv*)data;
	int state;

	// ST_1121
	// state = btread(BT848_I2C) & 0x02 ? 1 : 0;
	state = btread(TW6801_SBUSC) & 0x40 ? 1 : 0;	
	
	return state;
}

static int TW6801_bit_getsda(void *data)
{
	struct bttv *btv = (struct bttv*)data;
	int state;

	// ST_1121
	// state = btread(BT848_I2C) & 0x01;
//	state = btread(TW6801_SBUSC) & 0x01;	
	state = btread(TW6801_SBUSC) & 0x80 ? 1 : 0;	
	
	return state;
	
}

static struct i2c_algo_bit_data TW_i2c_algo_bit_template = {
	.setsda 	= TW6801_bit_setsda,
	.setscl  	= TW6801_bit_setscl,
	.getsda  	= TW6801_bit_getsda,
	.getscl  	= TW6801_bit_getscl,
	.udelay  	= 16,
// ST_DEL
// 	.mdelay 	= 10,
	.timeout 	= 200,
};

static struct i2c_adapter TW_i2c_adap_sw_template = {
	.owner             	= THIS_MODULE,
	.class             	= I2C_CLASS_TV_ANALOG,
	.name              = "TW6801",
	.id                	= I2C_HW_B_BT848,
	.client_register   = attach_inform,
};

/* ----------------------------------------------------------------------- */
/* I2C functions - hardware i2c                                            */

static int algo_control(struct i2c_adapter *adapter,
			unsigned int cmd, unsigned long arg)
{
	return 0;
}

static u32 functionality(struct i2c_adapter *adap)
{
	return I2C_FUNC_SMBUS_EMUL;
}

static int
TW6801_i2c_wait_done(struct bttv *btv)
{
	int rc = 0;

	/* timeout */
	if (wait_event_interruptible_timeout(btv->i2c_queue,
		btv->i2c_done, msecs_to_jiffies(85)) == -ERESTARTSYS)

	rc = -EIO;

	// ST_1121 for I2C control 
	// if (btv->i2c_done & BT848_INT_RACK)
	if (btv->i2c_done & TW6801_SBDONE)	{
		rc = 1;
	}		
	
	btv->i2c_done = 0;
	
	return rc;
	
}

// ST_1121 for I2C Control
/*
#define I2C_HW (BT878_I2C_MODE  | BT848_I2C_SYNC |\
		BT848_I2C_SCL | BT848_I2C_SDA)
*/

#define I2C_HW_WRITE ( TW6801_SB_WRITE  |TW6801_SSCLK | TW6801_SSDAT |TW6801_SB_HARDWARE )
#define I2C_HW_READ ( TW6801_SB_READ |TW6801_SSCLK | TW6801_SSDAT |TW6801_SB_HARDWARE )


static int
TW6801_i2c_sendbytes(struct bttv *btv, const struct i2c_msg *msg, int last)
{
	u32 xmit ;
	int retval, cnt ;

	u32	Length = 1 ;
	u32	SBCLK = 0x0A6 ;


	// Hardware I2C support
	if ( 0 == msg->len )
		return -EINVAL;
	
	/* start, address + first byte */
	// xmit = (msg->addr << 24) | I2C_HW_WRITE;
	
	xmit = 	(  msg->addr<< 25) | 
			I2C_HW_WRITE | 
			(Length <<16) | 
			(SBCLK<<8) ;
	
	btwrite(xmit, TW6801_SBUSC ) ;
	
	retval = TW6801_i2c_wait_done(btv) ;
	
	if (retval < 0)
		goto err ;
	
	if (retval == 0)
		goto eio ;

	for ( cnt = 0; cnt < msg->len; cnt++ ) {
		
		/* following bytes */
		xmit = msg->buf[cnt] ;

		DBG( "Count:%d, msg: %x\n", cnt,xmit ) ;
		
		btwrite( xmit, TW6801_SBUSSD ) ;

		retval = TW6801_i2c_wait_done(btv);
		
		if (retval < 0) goto err ;
		
		if (retval == 0) goto eio ;
		
	}
	
	return msg->len ;

eio:
	
	retval = -EIO;
	
err:
	
	return retval;

}


static int
bttv_i2c_sendbytes(struct bttv *btv, const struct i2c_msg *msg, int last)
{
	u32 xmit;
	int retval,cnt;

	/* sanity checks */
	if (0 == msg->len)
		return -EINVAL;

	/* start, address + first byte */
	xmit = (msg->addr << 25) | (msg->buf[0] << 16) | I2C_HW_WRITE ;
	
	if (msg->len > 1 || !last)
		xmit |= BT878_I2C_NOSTOP;
	
	btwrite(xmit, BT848_I2C);
	
	retval = TW6801_i2c_wait_done(btv);
	
	if (retval < 0)
		goto err;
	
	if (retval == 0)
		goto eio;

	// ST_1121 for I2C Control 
	/*
	if (i2c_debug) {
		
		DBG(" <W %02x %02x", msg->addr << 1, msg->buf[0]);
		
		if (!(xmit & BT878_I2C_NOSTOP))
			DBG(" >\n");
		
	}
	*/

	for (cnt = 1; cnt < msg->len; cnt++ ) {
		/* following bytes */
		xmit = (msg->buf[cnt] << 24) | I2C_HW_WRITE | BT878_I2C_NOSTART;
		if (cnt < msg->len-1 || !last)
			xmit |= BT878_I2C_NOSTOP;
		btwrite(xmit, BT848_I2C);
		retval = TW6801_i2c_wait_done(btv);
		if (retval < 0)
			goto err;
		if (retval == 0)
			goto eio;
		
		// ST_1121
		/*
		if (i2c_debug) {
			DBG(" %02x", msg->buf[cnt]);
			if (!(xmit & BT878_I2C_NOSTOP))
				DBG(" >\n");
		}
		*/
	}
	
	return msg->len;

 eio:
	retval = -EIO;
 err:
	if (i2c_debug)
		DBG(" ERR: %d\n",retval);
	return retval;
	
}




static int
bttv_i2c_readbytes(struct bttv *btv, const struct i2c_msg *msg, int last)
{
	u32 xmit;
	u32 cnt;
	int retval;

	for(cnt = 0; cnt < msg->len; cnt++) {
		xmit = (msg->addr << 25) | (1 << 24) | I2C_HW_READ ;
		if (cnt < msg->len-1)
			xmit |= BT848_I2C_W3B;
		if (cnt < msg->len-1 || !last)
			xmit |= BT878_I2C_NOSTOP;
		if (cnt)
			xmit |= BT878_I2C_NOSTART;
		btwrite(xmit, BT848_I2C);
		retval = TW6801_i2c_wait_done(btv);
		if (retval < 0)
			goto err;
		if (retval == 0)
			goto eio;
		msg->buf[cnt] = ((u32)btread(BT848_I2C) >> 8) & 0xff;
		if (i2c_debug) {
			if (!(xmit & BT878_I2C_NOSTART))
				DBG(" <R %02x", (msg->addr << 1) +1);
			DBG(" =%02x", msg->buf[cnt]);
			if (!(xmit & BT878_I2C_NOSTOP))
				DBG(" >\n");
		}
	}
	return msg->len;

 eio:
	retval = -EIO;
 err:
	if (i2c_debug)
		DBG(" ERR: %d\n",retval);
	return retval;
}



static int bttv_i2c_xfer(struct i2c_adapter *i2c_adap, struct i2c_msg *msgs, int num)
{
	struct bttv *btv = i2c_get_adapdata(i2c_adap);
	int retval = 0;
	int i;

	DBG( "----->bttv_i2c_xfer( num: %d ) \n", num ) ;

	/*
	if (i2c_debug)
		DBG("bt-i2c:");
	*/		

	// ST_1123 for I2C control
	// btwrite(BT848_INT_I2CDONE|BT848_INT_RACK, BT848_INT_STAT) ;
	
	for (i = 0 ; i < num; i++) {
		if (msgs[i].flags & I2C_M_RD) {
			/* read */
			retval = bttv_i2c_readbytes(btv, &msgs[i], i+1 == num);
			if (retval < 0)
				goto err;
		} else {
			/* write */
			// ST_1121 for I2c Control 
			// retval = bttv_i2c_sendbytes(btv, &msgs[i], i+1 == num);
			retval = TW6801_i2c_sendbytes(btv, &msgs[i], i+1 == num);			
			
			if (retval < 0)
				goto err;
			
		}
	}

	DBG( "<-----bttv_i2c_xfer(%d)\n", num ) ;
	
	return num;

 err:
	DBG( "<-----bttv_i2c_xfer(-1)\n") ; 	
	return retval;
	
}

static struct i2c_algorithm bttv_algo = {  //for HW
	.master_xfer   = bttv_i2c_xfer,

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,17)	
	.algo_control  = algo_control,
#endif

	.functionality = functionality,
};

static struct i2c_adapter TW_i2c_adap_hw_template = {
	.owner             = THIS_MODULE,
	.class         	= I2C_CLASS_TV_ANALOG,
	.name          	= "TW6801",
	.id            		= I2C_HW_B_BT848 /* FIXME */,
	.algo          	= &bttv_algo,
	.client_register = attach_inform,
};

/* ----------------------------------------------------------------------- */
/* I2C functions - common stuff                                            */

static int attach_inform(struct i2c_client *client)
{
	struct bttv *btv = i2c_get_adapdata(client->adapter);
	
	int addr=ADDR_UNSET;

	DBG( "----->attach_inform\n" ) ;


	if (ADDR_UNSET != bttv_tvcards[btv->c.type].tuner_addr)
		addr = bttv_tvcards[btv->c.type].tuner_addr;


	// ST_1121
	/*
	if (bttv_debug)
		DBG(KERN_DEBUG "bttv%d: %s i2c attach [addr=0x%x,client=%s]\n",
			btv->c.nr, client->driver->driver.name, client->addr,
			client->name);
	*/			
	
	if (bttv_debug)
		DBG(KERN_DEBUG "TW6801: %s i2c attach [addr=0x%x,client=%s]\n",
			client->driver->driver.name, client->addr,
			client->name);

	

	if (!client->driver->command)
		return 0;

	if (client->driver->id == I2C_DRIVERID_MSP3400)
		btv->i2c_msp34xx_client = client;
	
	if (client->driver->id == I2C_DRIVERID_TVAUDIO)
		btv->i2c_tvaudio_client = client;
	
	if (btv->tuner_type != UNSET) {
		struct tuner_setup tun_setup;

		if ((addr==ADDR_UNSET) ||
				(addr==client->addr)) {

			tun_setup.mode_mask = T_ANALOG_TV | T_DIGITAL_TV | T_RADIO;
			tun_setup.type = btv->tuner_type;
			tun_setup.addr = addr;
			bttv_call_i2c_clients(btv, TUNER_SET_TYPE_ADDR, &tun_setup);
		}

	}

	DBG( "<-----attach_inform\n" ) ;

	return 0;
	
}


void bttv_call_i2c_clients(struct bttv *btv, unsigned int cmd, void *arg)
{
	u32 xmit, Frequency ;
	struct v4l2_frequency *Fre = arg;	
	int retval ;
	u32 Bank, Divid1, Divid2 ;
	
	// Set the frequency to the tuner ...
	// I2C Device's Address 
	u32	Length = 4;
	u32	SBCLK = 0x0A6 ;


	if (0 != btv->i2c_rc)
		return;
	
	// ST_1121 for I2C Control 
	// i2c_clients_command(&btv->c.i2c_adap, cmd, arg);


	// ST_ADD_1123
	switch( cmd ){
		
		case VIDIOC_S_FREQUENCY :
		{

			Frequency = ( Fre->frequency )+0x02DC ;

			if( Frequency < 0x0D70 ) {
				Bank = 0x01 ;
			} else if ( Frequency > 0x0D10 && Frequency < 0x1F10 ) {
				Bank = 0x02 ;
			} else {
				Bank = 0x08 ;
			}

			xmit = 0 ;

			xmit |= 	( (0xC2>>1) << 25) | 
					TW6801_SB_WRITE  |
					TW6801_SB_HARDWARE | 
					TW6801_SSDAT |
					TW6801_SSCLK |
					(Length <<16) | 
					(SBCLK<<8) ;
			
			btwrite(xmit, TW6801_SBUSC ) ;

			if( Fre->type == V4L2_TUNER_RADIO ) {
				DBG( "=== RADIO ===\n" ) ;
				DBG( "Address==> %x \n", xmit ) ;
				DBG( "Tuner==> %x \n", Fre->tuner ) ;				
				
				// Frequency 
		
				
			} else {

				/*
				DBG( "=== ANALOG_TV ===\n" ) ;	
				DBG( "Address==> %x \n", xmit ) ;				
				DBG( "Frequency==> %x \n", Frequency ) ;					
				*/
				// Frequency 
				Divid1 = ( Frequency & 0xFF00 ) >> 8 ;
				Divid2 = Frequency & 0x00FF ;				

				xmit = 0 ;
				xmit |= Divid1 | (Divid2<<8) |(0xC6 << 16) | (Bank<<24)   ;
				/*
				printk( "DATA==> %x \n", xmit ) ;				
				printk( "===============\n" ) ;	
				*/
			}
			
	
			btwrite( xmit, TW6801_SBUSSD ) ;

			btwrite( 0x01, TW6801_SBUSTRIG ) ;
			
			retval = TW6801_i2c_wait_done(btv);

			// printk( "Write Frequency: %x, Retval: %d \n", xmit, retval ) ;				
			
			break ;

		}		
		case VIDIOC_G_FREQUENCY :

			/*
			Length = 0x01 ;	
			
			xmit =	( (0xC2>>1) << 25) | 
					TW6801_SB_WREN | 
					TW6801_SB_READ  |
					TW6801_SB_HARDWARE | 
					TW6801_SSDAT |
					TW6801_SSCLK |
					(Length <<20) | 
					(SBCLK<<8) ;

			printk( "Address==> %x \n", xmit ) ;

			btwrite( xmit, TW6801_SBUSC ) ;

			btwrite( 0x01, TW6801_SBUSTRIG ) ;

			// Frequency 
			xmit = Fre->frequency ;
			
			xmit = btread( TW6801_SBUSRD ) ;

			printk( "Read value: %x ...\n", xmit ) ;				
			*/
	
		default: 
			// printk( "Get unknow command ...\n" ) ;
			break ;
	}

	
}


static struct i2c_client TW_i2c_client_template = {
	.name	= "TW680x internal",
};


/* read I2C */
int bttv_I2CRead(struct bttv *btv, unsigned char addr, char *probe_for)
{
	unsigned char buffer = 0;

	if (0 != btv->i2c_rc)
		return -1;
	if (bttv_verbose && NULL != probe_for)
		DBG(KERN_INFO "bttv%d: i2c: checking for %s @ 0x%02x... ",
		       btv->c.nr,probe_for,addr);
	btv->i2c_client.addr = addr >> 1;
	if (1 != i2c_master_recv(&btv->i2c_client, &buffer, 1)) {
		if (NULL != probe_for) {
			if (bttv_verbose)
				DBG("not found\n");
		} else
			DBG(KERN_WARNING "bttv%d: i2c read 0x%x: error\n",
			       btv->c.nr,addr);
		return -1;
	}
	if (bttv_verbose && NULL != probe_for)
		DBG("found\n");
	return buffer;
}

/* write I2C */
int bttv_I2CWrite(struct bttv *btv, unsigned char addr, unsigned char b1,
		    unsigned char b2, int both)
{
	unsigned char buffer[2];
	int bytes = both ? 2 : 1;

	if (0 != btv->i2c_rc)
		return -1;
	btv->i2c_client.addr = addr >> 1;
	buffer[0] = b1;
	buffer[1] = b2;
	if (bytes != i2c_master_send(&btv->i2c_client, buffer, bytes))
		return -1;
	return 0;
}

/* read EEPROM content */
void __devinit bttv_readee(struct bttv *btv, unsigned char *eedata, int addr)
{
	memset(eedata, 0, 256);
	if (0 != btv->i2c_rc)
		return;
	btv->i2c_client.addr = addr >> 1;
	tveeprom_read(&btv->i2c_client, eedata, 256);
}

static char *i2c_devs[128] = {
	[ 0x1c >> 1 ] = "lgdt330x",
	[ 0x30 >> 1 ] = "IR (hauppauge)",
	[ 0x80 >> 1 ] = "msp34xx",
	[ 0x86 >> 1 ] = "tda9887",
	[ 0xa0 >> 1 ] = "eeprom",
	[ 0xc0 >> 1 ] = "tuner (analog)",
	[ 0xc2 >> 1 ] = "tuner (analog)",
};

static void do_i2c_scan(char *name, struct i2c_client *c)
{
	unsigned char buf;
	int i,rc;

	DBG( "----->do_i2c_scan-0224\n" ) ;

	// ST_0223, for suse load error	
	// for (i = 0; i < 128; i++) {
	for (i = 0; i < ARRAY_SIZE(i2c_devs); i++) {

		c->addr = i;
		rc = i2c_master_recv(c,&buf,0);
		if (rc < 0)
			continue;
		DBG("%s: i2c scan: found device @ 0x%x  [%s]\n",
		       name, i << 1, i2c_devs[i] ? i2c_devs[i] : "???");
	}

	DBG( "<-----do_i2c_scan\n" ) ;
	
}



/* init + register i2c algo-bit adapter */
int __devinit init_tw_i2c(struct bttv *btv)
{

	DBG( "----->init_tw_i2c \n" ) ;

	// ST_ADD_0924 for tw6801 support, fix the hardware I2C support 
	//i2c_hw = 1 ;
	// ST_0224, for suse load driver failed 
	// i2c_scan = 1 ;
	i2c_scan = 0 ;
	//

	memcpy(&btv->i2c_client, &TW_i2c_client_template,
	       sizeof(TW_i2c_client_template));

	if (i2c_hw)
		btv->use_i2c_hw = 1 ;
	
	if (btv->use_i2c_hw) {
		
		/* bt878 */
		memcpy(&btv->c.i2c_adap, &TW_i2c_adap_hw_template,
		       sizeof(TW_i2c_adap_hw_template));
		
	} else {
	
		/* bt848 */
		memcpy(&btv->c.i2c_adap, &TW_i2c_adap_sw_template,
		       sizeof(TW_i2c_adap_sw_template));
		
		memcpy(&btv->i2c_algo, &TW_i2c_algo_bit_template,
		       sizeof(TW_i2c_algo_bit_template));
		
		btv->i2c_algo.data = btv;
		
		btv->c.i2c_adap.algo_data = &btv->i2c_algo;
		
	}


	btv->c.i2c_adap.dev.parent = &btv->c.pci->dev;
	
	snprintf(btv->c.i2c_adap.name, sizeof(btv->c.i2c_adap.name),
		 "bt%d #%d [%s]", btv->id, btv->c.nr,
		 btv->use_i2c_hw ? "hw" : "sw");

	i2c_set_adapdata(&btv->c.i2c_adap, btv);

	DBG( "i2c_set_adapdata is done ...\n" ) ;
	
	btv->i2c_client.adapter = &btv->c.i2c_adap ;

	if (bttv_tvcards[btv->c.type].no_video)
		btv->c.i2c_adap.class &= ~I2C_CLASS_TV_ANALOG;
	
	if (bttv_tvcards[btv->c.type].has_dvb)
		btv->c.i2c_adap.class |= I2C_CLASS_TV_DIGITAL;

	if (btv->use_i2c_hw) {
		btv->i2c_rc = i2c_add_adapter(&btv->c.i2c_adap);
	} else {
		TW6801_bit_setscl(btv,1);
		TW6801_bit_setsda(btv,1);
		btv->i2c_rc = i2c_bit_add_bus(&btv->c.i2c_adap);
	}
	
	if (0 == btv->i2c_rc && i2c_scan) {
		do_i2c_scan(btv->c.name,&btv->i2c_client);
	}

	DBG( "<-----init_tw_i2c \n" ) ;
	
	return btv->i2c_rc;
	
}

int __devexit fini_bttv_i2c(struct bttv *btv)
{
	if (0 != btv->i2c_rc)
		return 0;

	// ST_0725
	// if (btv->use_i2c_hw) {
		return i2c_del_adapter(&btv->c.i2c_adap);
	
	// ST_0725
	// } else {
	//	return i2c_bit_del_bus(&btv->c.i2c_adap);
	// }
}

/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
