/*
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <asm/page.h>
#include <asm/pgtable.h>

#include <media/v4l2-ioctl.h>

#include "bttvp.h"

// ST_ADD_0925
#include "techwell.h"

// ST_0306
extern int switching ;


#define 	VCR_HACK_LINES		4

#define	EnableDMARetry 		50

/* ---------------------------------------------------------- */
/* risc code generators                                       */

int
tw_dma_packed
(
	struct bttv *btv, 
	struct btcx_riscmem *risc,
	struct scatterlist *sglist,
	unsigned int offset, 
	unsigned int bpl,
	unsigned int padding, 
	unsigned int lines
)
{
	u32 instructions,line,todo;
	struct scatterlist *sg;
	u32 *rp;
	int rc;

	// ST_1030
	int	remind ;


	DBG_MMAP("---> tw_dma_packed - #%d  \n", btv->c.nr ) ;

	/* estimate risc mem: worst case is one write per page border +
	   one write per scan line + sync + jump (all 2 dwords).  padding
	   can cause next bpl to start close to a page border.  First DMA
	   region may be smaller than PAGE_SIZE */
	instructions  = 1 + ((bpl + padding) * lines) / PAGE_SIZE + lines;
	instructions += 2;

	// DBG( "   --->tw_dma_packed: Address: %x, top_irq: %d, irqflags: %d \n", *risc, btv->curr.top_irq, btv->curr.irqflags ) ;

	if ((rc = btcx_riscmem_alloc(btv->c.pci,risc,instructions*8)) < 0) {
		DBG_MMAP("---> btcx_riscmem_alloc < 0 !!!! \n") ;
		return rc;
	}

	// ST_0423
	//spin_lock(&btv->s_lock);


	/* sync instruction */
	rp = risc->cpu ;

	//DBG( "--->tw_dma_packed: bpl: %d, padding: %d, line: %d, Offset: %d, top_irq: %d, irqflags: %d, instr: %d \n", 
	//		bpl, padding, lines, offset, btv->curr.top_irq, btv->curr.irqflags, instructions ) ;

	// ST_0725
	// *(rp++) = cpu_to_le32( TW6801_DMA_TYPE_SYNCO ) ; 	
	
	// ST_1112
	// if( btv->curr.irqflags ) {

	// ST_0508_DEL
#if 0	
	if( offset > 0 ) {		
		DBG("---> Create RISC: %x program for EVEN field  !!!! \n", risc ) ;
		*(rp++) = cpu_to_le32( TW6801_DMA_TYPE_SYNCE ) ; 	
	} else {
		DBG("---> Create RISC: %x program for ODD field  !!!! \n", risc ) ;
		*(rp++) = cpu_to_le32( TW6801_DMA_TYPE_SYNCO ) ; 	
	}

	
	*(rp++) = cpu_to_le32(0);
#endif

	/* scan lines */
	sg = sglist;
	
	for (line = 0; line < lines; line++) {

		// DBG_DMA( "-->Line: %d,", line ) ;

		// ST_0112
		if ( (btv->opt_vcr_hack) && ( line >= (lines - VCR_HACK_LINES)))
			continue;

		while (offset && offset >= sg_dma_len(sg)) {
			offset -= sg_dma_len(sg);
			sg++;
		}

		// DBG_DMA( "%d ->offset: %d, dma_len: %d", lines, offset, sg_dma_len(sg) ); 
		
		if ( bpl <= sg_dma_len(sg)-offset ) {
			
			/* fits into current chunk */
			// DBG_DMA( "->1(offset: %d) ", offset  ) ;			

			*(rp++)=cpu_to_le32( TW6801_DMA_TYPE_LINESTART |bpl ) ;			
			*(rp++)=cpu_to_le32( sg_dma_address(sg)+offset ) ;
			offset += bpl ;

		} else {
		
			/* scanline needs to be splitted */
			todo = bpl ;

			*(rp++)=cpu_to_le32( TW6801_DMA_TYPE_LINESTART |(sg_dma_len(sg)-offset)) ;			
			*(rp++)=cpu_to_le32( (sg_dma_address(sg)+offset) ) ;
			todo -= (sg_dma_len(sg)-offset);
			remind =(sg_dma_len(sg)-offset); 
			offset = 0 ;
			sg++;

			while ( todo > sg_dma_len(sg) ) {				

				// DBG( "Packed-3----, "  ) ;							
	
				// ST_1113
				*(rp++)=cpu_to_le32( TW6801_DMA_TYPE_LINESTART |sg_dma_len(sg));				
				// *(rp++) = cpu_to_le32( TW6801_DMA_TYPE_INLINE | sg_dma_len(sg) ) ;

				*(rp++) = cpu_to_le32( sg_dma_address(sg) ) ;
				
				todo -= sg_dma_len(sg);
				sg++;
				
			}

			// ST_0725
			*(rp++)=cpu_to_le32( TW6801_DMA_TYPE_INLINE |(remind<<12)|todo ) ;			

			*(rp++)=cpu_to_le32(sg_dma_address(sg));

			offset += todo;
			
		}
		
		offset += padding;

		// DBG_DMA( "->offset: %d \n", offset  ) ;			
		
	}

	// DBG( " <-- \n "  ) ;			

	/* save pointer to jmp instruction address */
	risc->jmp = rp ;

	/* ST_ADD_072409 */
	*(rp++) = cpu_to_le32( TW6801_DMA_TYPE_JUMP ) ; 
	// *(rp++) = cpu_to_le32( TW6801_DMA_TYPE_JUMP | TW6801_DMA_IRQ  ) ; 
	

	if( offset > 0 ) {		
		// DBG("---> This is EVEN field  !!!! \n" ) ;
		*(rp++) = cpu_to_le32( btv->main.dma + (12<<2) ) ; 	
	} else {
		// DBG("---> This is  ODD field  !!!! \n" ) ;
		*(rp++) = cpu_to_le32( btv->main.dma + (8<<2) ) ; 	
	}


	// ST_0423
	//spin_unlock(&btv->s_lock);

	// DBG_MMAP("<--- tw_dma_packed - #%d  \n", btv->c.nr ) ;
	
	BUG_ON((risc->jmp - risc->cpu + 2) * sizeof(*risc->cpu) > risc->size);
	
	return 0;

	
}



static int
tw_risc_planar
(
	struct bttv *btv, 
	struct btcx_riscmem *risc,
	struct scatterlist *sglist,
	unsigned int yoffset,  
	unsigned int ybpl,
	unsigned int ypadding, 
	unsigned int ylines,
	unsigned int uoffset,  
	unsigned int voffset,
	unsigned int hshift,   
	unsigned int vshift,
	unsigned int cpadding
)
{
	unsigned int instructions,line,todo,ylen,chroma;
	u32 *rp,ri;
	struct scatterlist *ysg;
	struct scatterlist *usg;
	struct scatterlist *vsg;

	// ST_1015
	// int topfield = (0 == yoffset);

	int rc;

	DBG_SWITCH( "----->tw_risc_planar \n" ) ;

	/* estimate risc mem: worst case is one write per page border +
	   one write per scan line (5 dwords)
	   plus sync + jump (2 dwords) */
	instructions  = (ybpl * ylines * 2) / PAGE_SIZE + ylines;
	instructions += 2;
	if ((rc = btcx_riscmem_alloc(btv->c.pci,risc,instructions*4*5)) < 0)
		return rc;

	/* sync instruction */
	rp = risc->cpu;

	// ST_MODIFY_0925 for dma control
	// *(rp++) = cpu_to_le32(BT848_RISC_SYNC|BT848_FIFO_STATUS_FM3);
	// ST_1101
	// *(rp++) = cpu_to_le32( TW6801_DMA_TYPE_SYNCO ) ;

	// ST_0126_DEL
	/*
	if( btv->curr.irqflags ) {
		*(rp++) = cpu_to_le32( TW6801_DMA_TYPE_SYNCE ) ; 	
	} else {
		*(rp++) = cpu_to_le32( TW6801_DMA_TYPE_SYNCO ) ; 	
	}
	*/

	
	*(rp++) = cpu_to_le32(0);

	/* scan lines */
	ysg = sglist;
	usg = sglist;
	vsg = sglist;
	for (line = 0; line < ylines; line++) {
		if ((btv->opt_vcr_hack) &&
		    (line >= (ylines - VCR_HACK_LINES)))
			continue;

		// ST_1015
		/*
		switch (vshift) {
		case 0:
			chroma = 1;
			break;
		case 1:
			if (topfield)
				chroma = ((line & 1) == 0);
			else
				chroma = ((line & 1) == 1);
			break;
		case 2:
			if (topfield)
				chroma = ((line & 3) == 0);
			else
				chroma = ((line & 3) == 2);
			break;
		default:
			chroma = 0;
			break;
		}
		*/

		switch (vshift) {
		case 0:  chroma = 1;           break;
		case 1:  chroma = !(line & 1); break;
		case 2:  chroma = !(line & 3); break;
		default: chroma = 0;
		}
	

		for (todo = ybpl; todo > 0; todo -= ylen) {
			/* go to next sg entry if needed */
			while (yoffset && yoffset >= sg_dma_len(ysg)) {
				yoffset -= sg_dma_len(ysg);
				ysg++;
			}
			while (uoffset && uoffset >= sg_dma_len(usg)) {
				uoffset -= sg_dma_len(usg);
				usg++;
			}
			while (voffset && voffset >= sg_dma_len(vsg)) {
				voffset -= sg_dma_len(vsg);
				vsg++;
			}

			/* calculate max number of bytes we can write */
			ylen = todo;
			if (yoffset + ylen > sg_dma_len(ysg))
				ylen = sg_dma_len(ysg) - yoffset;
			if (chroma) {
				if (uoffset + (ylen>>hshift) > sg_dma_len(usg))
					ylen = (sg_dma_len(usg) - uoffset) << hshift;
				if (voffset + (ylen>>hshift) > sg_dma_len(vsg))
					ylen = (sg_dma_len(vsg) - voffset) << hshift;
				
				// ST_1101
				// ri = BT848_RISC_WRITE123;
				ri = TW6801_DMA_TYPE_LINESTART ;


			} else {

				// ST_1101
				// ri = BT848_RISC_WRITE1S23;
				ri = TW6801_DMA_TYPE_INLINE;

			}
			if (ybpl == todo)
				// ST_1101
				// ri |= BT848_RISC_SOL;
				ri |= TW6801_DMA_TYPE_INLINE;


			if (ylen == todo)
				// ST_1101
				// ri |= BT848_RISC_EOL;
				ri |= TW6801_DMA_TYPE_JUMP;


			/* write risc instruction */
			// ST_1101	
			// *(rp++)=cpu_to_le32(ri | ylen);
			*(rp++)=cpu_to_le32(ri|((ylen >> hshift) << 12 /*16*/) |
					    (ylen >> hshift));
			*(rp++)=cpu_to_le32(sg_dma_address(ysg)+yoffset);

			yoffset += ylen;
			if (chroma) {
				*(rp++)=cpu_to_le32(sg_dma_address(usg)+uoffset);
				uoffset += ylen >> hshift;
				*(rp++)=cpu_to_le32(sg_dma_address(vsg)+voffset);
				voffset += ylen >> hshift;
			}
		}
		yoffset += ypadding;
		if (chroma) {
			uoffset += cpadding;
			voffset += cpadding;
		}
	}

	/* save pointer to jmp instruction address */
	risc->jmp = rp;
	BUG_ON((risc->jmp - risc->cpu + 2) * sizeof(*risc->cpu) > risc->size);

	DBG( "<-----tw_risc_planar \n" ) ;
	
	return 0;
}



static int
bttv_risc_overlay
(
	struct bttv *btv, 
	struct btcx_riscmem *risc,
	const struct bttv_format *fmt, 
	struct bttv_overlay *ov,
	int skip_even, int skip_odd
)
{
	int dwords,rc,line,maxy,start,end,skip,nskips;
	struct btcx_skiplist *skips;
	u32 *rp,ri,ra;
	u32 addr;

	DBG_SWITCH( "----->bttv_risc_overlay\n" ) ;


	/* skip list for window clipping */
	if ( NULL == (skips = kmalloc(sizeof(*skips) * ov->nclips,GFP_KERNEL)) ) {
		return -ENOMEM ;
	}
	
	/* estimate risc mem: worst case is (1.5*clip+1) * lines instructions
	   + sync + jump (all 2 dwords) */
	dwords  = (3 * ov->nclips + 2) *
		((skip_even || skip_odd) ? (ov->w.height+1)>>1 :  ov->w.height) ;
	
	dwords += 4 ;

	if ((rc = btcx_riscmem_alloc(btv->c.pci,risc,dwords*4)) < 0) {
		kfree(skips) ;
		return rc ;
	}

	/* sync instruction */
	rp = risc->cpu ;

	*(rp++) = cpu_to_le32(TW6801_DMA_TYPE_LINESTART) ;

	*(rp++) = cpu_to_le32(0) ;

	addr  = (unsigned long)btv->fbuf.base ;
	
	addr += btv->fbuf.fmt.bytesperline * ov->w.top ;
	
	addr += (fmt->depth >> 3) * ov->w.left ;

	/* scan lines */
	for ( maxy = -1, line = 0; line < ov->w.height; 
		line++, addr += btv->fbuf.fmt.bytesperline ) {
		 	
		if ( (btv->opt_vcr_hack) && (line >= (ov->w.height - VCR_HACK_LINES)) ) {
			continue ;
		}
		
		if ( (line%2) == 0  &&  skip_even ) {
			continue ;
		}
		
		if ( (line%2) == 1  &&  skip_odd ) {
			continue ;
		}

		/* calculate clipping */
		if ( line > maxy )
			btcx_calc_skips(line, ov->w.width, &maxy,
					skips, &nskips, ov->clips, ov->nclips) ;

		/* write out risc code */
		for ( start = 0, skip = 0; start < ov->w.width; start = end ) {
			
			if ( skip >= nskips ) {

				ri  = TW6801_DMA_TYPE_SYNCO  ;
				end = ov->w.width ;
				
			} else if (start < skips[skip].start) {

				ri  = TW6801_DMA_TYPE_SYNCE ;
				end = skips[skip].start ;
				
			} else {

				ri  = TW6801_DMA_TYPE_JUMP ;
				end = skips[skip].end ;
				skip++ ;
				
			}

			if ( TW6801_DMA_TYPE_SYNCO == ri ) {
				
				ra = addr + (fmt->depth>>3)*start ;
				
			} else {
			
				ra = 0 ;
			
			}


			if ( 0 == start ) {
				ri |= TW6801_DMA_TYPE_LINESTART ;
			}


			if ( ov->w.width == end ) {
				ri |= TW6801_DMA_TYPE_JUMP ;
			}

			ri |= (fmt->depth>>3) * (end-start) ;

			*(rp++)=cpu_to_le32(ri) ;
			
			if (0 != ra) {
				*(rp++) = cpu_to_le32(ra) ;
			}
				
		}
		 
	}



	/* save pointer to jmp instruction address */
	risc->jmp = rp ;
	
	BUG_ON((risc->jmp - risc->cpu + 2) * sizeof(*risc->cpu) > risc->size) ;
	
	kfree(skips);

	DBG( "<-----bttv_risc_overlay\n" ) ;
	
	return 0 ;
	
}

/* ---------------------------------------------------------- */
static void
bttv_calc_geo
(
	struct bttv *btv, 
	struct bttv_geometry *geo,
	int width, 
	int height, 
	int interleaved, 
	int norm
)
{
	const struct bttv_tvnorm *tvnorm = &bttv_tvnorms[norm];
	u32 xsf, sr;
	int vdelay;

	int 	swidth       = tvnorm->swidth;
	int 	totalwidth   = tvnorm->totalwidth;
	int 	scaledtwidth = tvnorm->scaledtwidth;
	int	sheight 		= tvnorm->sheight ;

	DBG("----->bttv_calc_geo\n" ) ;
	

	if (bttv_tvcards[btv->c.type].muxsel[btv->input] < 0) {
		swidth       = 720;
		totalwidth   = 858;
		scaledtwidth = 858;
	}

	// ST_0329
	if( !interleaved ) {
		sheight = sheight /2 ;
	}

	vdelay = tvnorm->vdelay;
	geo->hdelay =  tvnorm->hdelayx1 ;

	// ST_0508
	// geo->hdelay = ( tvnorm->hdelayx1 * width ) / 720 ;
	geo->hdelay = tvnorm->hdelayx1  ;

	// geo->hdelay = ((geo->hdelay*width)/swidth)/2; // (totalwidth-swidth)/4;

	geo->vdelay = vdelay ;
	geo->width = swidth ;
	geo->sheight = sheight ;

	// ST_TEST_0115
	//geo->crop = ( ( swidth >>8)&0x03 ) |(  ( ( sheight/2)>>4)&0x30 ) ;
	// ST_0402
	geo->crop = ( ( swidth >>8)&0x03 ) |(  ( ( sheight)>>4)&0x30 ) ;	

	DBG( "bttv_calc_geo---> Type:%d, Interleaved: %d, sw:%d, sh: %d, tw:%d, scw:%d, vd:%d, hd:%d, w:%d, h:%d, crop:%x \n", 
			btv->c.type,
			interleaved, 
			swidth,
			sheight,
			totalwidth,
			scaledtwidth,
			vdelay,
			geo->hdelay,
			width,
			height,
			geo->crop ) ;

	geo->hscale =  ( swidth * 256 )  /width  ;


	geo->vscale =  ( sheight *256 ) / height  ; 

	DBG( "bttv_calc_geo: hscale: %x, vscale: %x ....\n ", geo->hscale, geo->vscale ) ;

	DBG( "<-----bttv_calc_geo\n" ) ;
	
}

static void
bttv_apply_geo
(
	struct bttv *btv, 
	struct bttv_geometry *geo, 
	int odd
)
{

	u16		TempScaleHI = 0 ;
	u16		TempHeight = 0 ;

	DBG( "----->bttv_apply_geo\n" ) ;


	// ST_DEL_1003
#if 0
	if (geo->comb)
		btor(BT848_VSCALE_COMB, BT848_E_VSCALE_HI+off);
	else
		btand(~BT848_VSCALE_COMB, BT848_E_VSCALE_HI+off);

	btwrite(geo->vtc,             BT848_E_VTC+off);
	btwrite(geo->hscale >> 8,     BT848_E_HSCALE_HI+off);
	btwrite(geo->hscale & 0xff,   BT848_E_HSCALE_LO+off);
	btaor((geo->vscale>>8), 0xe0, BT848_E_VSCALE_HI+off);
	btwrite(geo->vscale & 0xff,   BT848_E_VSCALE_LO+off);
	btwrite(geo->width & 0xff,    BT848_E_HACTIVE_LO+off);
	btwrite(geo->hdelay & 0xff,   BT848_E_HDELAY_LO+off);
	btwrite(geo->sheight & 0xff,  BT848_E_VACTIVE_LO+off);
	btwrite(geo->vdelay & 0xff,   BT848_E_VDELAY_LO+off);
	btwrite(geo->crop,            BT848_E_CROP+off);
	btwrite(geo->vtotal>>8,       BT848_VTOTAL_HI);
	btwrite(geo->vtotal & 0xff,   BT848_VTOTAL_LO);
#else

	// DBG( "bttv_apply_geo: hscale: %x, vscale: %x ....\n ", geo->hscale, geo->vscale ) ;

	TempScaleHI = (geo->hscale >> 8 ) ;
	TempScaleHI |= ( (geo->vscale >> 4 ) & 0xF0 ) ;

	DBG( "bttv_apply_geo2: hscale: %x, vscale: %x, scaleHI: %x, hdelay: %x, vdelay: %x, Crop: %x, HActive: %x ....\n ", 
		geo->hscale, geo->vscale, TempScaleHI, geo->hdelay , geo->vdelay, geo->crop, geo->width) ;

		// TempHeight  = geo->sheight / 2 ;

		btwrite(geo->hscale & 0xFF,   TW6801_HSCALE_LO);
		btwrite(geo->vscale & 0xFF,   TW6801_VSCALE_LO);
		btwrite( TempScaleHI ,   TW6801_SCALE_HI );
		btwrite(geo->hdelay & 0xFF,   TW6801_HDELAY_LO);
		btwrite(geo->vdelay & 0xFF,   TW6801_VDELAY_LO);

		
		// ST_0508_DEL
		// btwrite(geo->width & 0xFF,   TW6801_HACTIVE_LO);
		//btwrite(geo->crop & 0xFF,   TW6801_CROP_HI );
		
		btwrite(geo->hscale & 0xFF,   TW6801_H2SCALE_LO);
		btwrite(geo->vscale & 0xFF,   TW6801_V2SCALE_LO);
		btwrite( TempScaleHI ,   TW6801_2SCALE_HI );
		btwrite(geo->hdelay& 0xFF,   TW6801_HDELAY_LO2);
		btwrite(geo->vdelay& 0xFF,   TW6801_VDELAY_LO2);

		// ST_0508_DEL
		// btwrite(geo->width & 0xFF,   TW6801_HACTIVE_LO2);
		// btwrite(geo->crop & 0xFF,   TW6801_CROP_HI2 );

#endif


	DBG( "<-----bttv_apply_geo\n" ) ;
	
}



// ST_0125
void
tw_EnableDMA
(
	struct bttv *btv
)
{
	struct timeval ts ;
	unsigned long timeout;	

	DBG( "----->tw_EnableDMA - %d \n", btv->c.nr ) ; 			

	if( btv->m_EnableRISC == 1) {
		DBG( "<-----DMA - %d always enable .... \n", btv->c.nr ) ;
		return ;
	}

	DBG( "Set start address: %x...\n", btv->main.dma ) ;

	btwrite( btv->main.dma, TW6801_VDMAP_SA ) ;

	btv->m_EnableRISC = 1 ;

	if( switching ) {

		// timeout = jiffies + ( msecs_to_jiffies( btv->m_LoseSignalTimer ) ) ;
		timeout = jiffies + TW6801_TIMEOUT ;
		btv->RISCTimer.expires = timeout ;

		mod_timer(&btv->RISCTimer, timeout ) ;		
	
	}

	btor( (TW6801_VDMAC_VDMAP_EN|TW6801_VDMAC_VFIFO_EN), TW6801_VDMAC );	

	DBG_SWITCH( "<-----tw_EnableDMA - %d \n", btv->c.nr ) ; 

	return ;
	
}

void
tw_DisableDMA
(
	struct bttv *btv
)
{

	struct timeval ts ;

	do_gettimeofday(&ts);

	DBG_SWITCH( "----->tw_DisableDMA - %d, time: %d \n", btv->c.nr, ts.tv_usec ) ; 			

	btand( ~(TW6801_VDMAC_VDMAP_EN), TW6801_VDMAC );	

	btv->m_EnableRISC = 0 ;

	DBG_SWITCH( "<-----tw_DisableDMA - %d \n",  btv->c.nr ) ; 

	return ;
	
}



int
tw_dma_init_main(struct bttv *btv)
{
	int rc;

	DBG_SWITCH( "----->tw_dma_init_main \n" ) ;
		
	if ((rc = btcx_riscmem_alloc(btv->c.pci,&btv->main,PAGE_SIZE )) < 0) {
		DBG( "<-----tw_dma_init_main-1 \n" ) ;	
		return rc;
	}


	DBG_SWITCH( "TW6801-%d: DMA main @ %08Lx\n", btv->c.nr,(unsigned long long)btv->main.dma ) ;
	
	// ST_NOTE_0924 PAGE_SIZE = 4096 

	btv->main.cpu[0] = cpu_to_le32( TW6801_DMA_TYPE_SYNCO ) ;	
	btv->main.cpu[1] = cpu_to_le32(0);

	btv->main.cpu[2] = cpu_to_le32( TW6801_DMA_TYPE_JUMP ) ;	
	btv->main.cpu[3] = cpu_to_le32(btv->main.dma + (4<<2));


	/* top field */
	btv->main.cpu[4] = cpu_to_le32( TW6801_DMA_TYPE_JUMP );	
	btv->main.cpu[5] = cpu_to_le32(btv->main.dma + (6<<2));

	btv->main.cpu[6] = cpu_to_le32( TW6801_DMA_TYPE_JUMP ) ;	
	btv->main.cpu[7] = cpu_to_le32(btv->main.dma + (8<<2));


	// 
	// ST_11212008_TEST, I just try to make the RISC for ODD only ...
	btv->main.cpu[8] = cpu_to_le32( TW6801_DMA_TYPE_SYNCE )  ;
	btv->main.cpu[9] = cpu_to_le32(0);	

	/* bottom field */
	btv->main.cpu[10] = cpu_to_le32( TW6801_DMA_TYPE_JUMP ) ;

	btv->main.cpu[11] = cpu_to_le32( btv->main.dma + (12<<2) ) ;

	btv->main.cpu[12] = cpu_to_le32( TW6801_DMA_TYPE_JUMP ) ;
	btv->main.cpu[13] = cpu_to_le32( btv->main.dma + (14<<2) ) ;

	// Jump back to TOP
	// ST_ADD_0330, For some application initialize problem ...
	btv->main.cpu[14] = cpu_to_le32( TW6801_DMA_TYPE_JUMP | TW6801_DMA_IRQ ) ;		


	// ST_0508_CHK
#if 0
	btv->main.cpu[15] = cpu_to_le32(btv->main.dma + (0<<2));

#else
	btv->main.cpu[15] = cpu_to_le32(btv->main.dma + (18<<2));
	btv->main.cpu[16] = cpu_to_le32( TW6801_DMA_TYPE_DUMMY) ;		
	btv->main.cpu[17] = cpu_to_le32(0);
	btv->main.cpu[18] = cpu_to_le32( TW6801_DMA_TYPE_DUMMY) ;		
	btv->main.cpu[19] = cpu_to_le32(0);

	/*ST_ADD_073009 */
	int i = 0 ;
	for( i = 20; i < 1000; i +=2 ) {	
		btv->main.cpu[i] = cpu_to_le32( TW6801_DMA_TYPE_DUMMY) ;		
		btv->main.cpu[i+1] = cpu_to_le32(0);
	}

	
	/* ST_0129*/	
	// btv->main.cpu[40] = cpu_to_le32( TW6801_DMA_TYPE_JUMP) ;		
	btv->main.cpu[1000] = cpu_to_le32( TW6801_DMA_TYPE_JUMP) ;		

	/* ST_0129 */
	// btv->main.cpu[41] = cpu_to_le32(btv->main.dma + (16<<2)) ;
	btv->main.cpu[1001] = cpu_to_le32(btv->main.dma + (16<<2)) ;

	//
#endif	

	DBG_SWITCH( "<-----tw_dma_init_main \n" ) ;	
	
	return 0;
	
}




int
tw_dma_hook
(
	struct bttv	*btv, 
	int 			slot, 
	struct btcx_riscmem *risc,
   	int 			irqflags
)
{
	unsigned long cmd;
	unsigned long next = btv->main.dma + ((slot+2) << 2);

	// ST_0725
	unsigned int Temp = 0 ;


	DBG("---> tw_dma_hook - # \%d  \n", btv->c.nr ) ;
	

	// ST_0423
	//spin_lock(&btv->s_lock);

	// if( btv->c.nr == DBG_DeviceNumber  )
	//DBG_SWITCH( "----->tw_dma_hook-%d\n",  btv->c.nr ) ;

	if ( NULL == risc ) {
		
		//if( btv->c.nr == DBG_DeviceNumber  ) 	
		//	DBG_MMAP( ">>>>>>> Risc=%p slot[%d]=NULL <<<<<<< \n", btv->c.nr,risc,slot);
		
		btv->main.cpu[slot+1] = cpu_to_le32(next);
		
	} else {

		/*	
		if( btv->c.nr == DBG_DeviceNumber  ) {
			DBG_SWITCH( "Risc=%p slot[%d]=%08Lx irq=%d\n",  
				risc,
				slot,
				(unsigned long long)risc->dma,
				irqflags ) ;
		}
		*/
	
		cmd = TW6801_DMA_TYPE_JUMP ;		

		Temp = 8 ;

		// ST_0507_TEST
		/*
		if ( irqflags ) {
			cmd |= TW6801_DMA_IRQ ;			
		}
		*/

		/* ST_DEL_072409 */
		risc->jmp[0] = cpu_to_le32(cmd);
		risc->jmp[1] = cpu_to_le32(next);

		// ST_1015
		// btv->main.cpu[slot+1] = cpu_to_le32(risc->dma);

		// ST_0417, This point will make the screen greening.
// ST_0423
#if 1
		// ST_0508_TEST
#if 1
		btv->main.cpu[slot+1] = cpu_to_le32(risc->dma);	

#else
		btv->main.cpu[slot+1] = cpu_to_le32(risc->dma+Temp);	
#endif

#else
		if( slot == RISC_SLOT_O_FIELD ) {
			//DBG_FPS( ">> RISC_SLOT_O_FIELD-%d : dma: %p <<  \n", btv->c.nr, risc->dma);
			//btv->main.cpu[slot+1] = cpu_to_le32(risc->dma+ (8<<2));	
		} else {
			//DBG_FPS( ">> RISC_SLOT_E_FIELD-%d : dma: %p <<  \n", btv->c.nr, risc->dma);	
			//btv->main.cpu[21] = cpu_to_le32(risc->dma) ;		
		}
#endif
		
		
	}

	//DBG_SWITCH( "<-----tw_dma_hook-%d\n",  btv->c.nr ) ;

	// ST_0423
	// spin_unlock(&btv->s_lock);


	DBG("<--- tw_dma_hook - # %d  \n", btv->c.nr ) ;
	
	return 0;
	
}




// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
void
tw_dma_free
(
	struct videobuf_queue *q,
	struct bttv *btv, 
	struct bttv_buffer *buf
)
{
	struct videobuf_dmabuf *dma=videobuf_to_dma(&buf->vb);

	BUG_ON(in_interrupt());



	DBG_FPS( "---->tw_dma_free-%d\n", btv->c.nr ) ;
	
	videobuf_waiton(&buf->vb,0,0);

	DBG_FPS( "<---videobuf_waiton-%d\n", btv->c.nr ) ;

	videobuf_dma_unmap(q, dma);
	videobuf_dma_free(dma);
	btcx_riscmem_free(btv->c.pci,&buf->bottom);
	btcx_riscmem_free(btv->c.pci,&buf->top);
	buf->vb.state = VIDEOBUF_NEEDS_INIT;
}

#else
void
tw_dma_free
(
	struct videobuf_queue *q,
	struct bttv *btv, 
	struct bttv_buffer *buf
)
{
	BUG_ON(in_interrupt());
	videobuf_waiton(&buf->vb,0,0);
	videobuf_dma_unmap(q, &buf->vb.dma);
	videobuf_dma_free(&buf->vb.dma);
	btcx_riscmem_free(btv->c.pci,&buf->bottom);
	btcx_riscmem_free(btv->c.pci,&buf->top);
	buf->vb.state = STATE_NEEDS_INIT;
}

#endif


int
bttv_buffer_activate_vbi(struct bttv *btv,
			 struct bttv_buffer *vbi)
{
	/* vbi capture */
	if (vbi) {

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
		vbi->vb.state = VIDEOBUF_ACTIVE;
#else
		vbi->vb.state = STATE_ACTIVE;
#endif
		list_del(&vbi->vb.queue);
		tw_dma_hook(btv, RISC_SLOT_O_VBI, &vbi->top,    0);
		tw_dma_hook(btv, RISC_SLOT_E_VBI, &vbi->bottom, 4);
	} else {
		tw_dma_hook(btv, RISC_SLOT_O_VBI, NULL, 0);
		tw_dma_hook(btv, RISC_SLOT_E_VBI, NULL, 0);
	}
	return 0;
}


/*ST_0727_ADD*/
void
tw_buffer_reactive_video
(
	struct bttv *btv,
	struct bttv_buffer_set *set
)
{

	if ( NULL != set->top  &&  NULL != set->bottom ) {
		if (set->top == set->bottom) {
			set->top->vb.state = STATE_ACTIVE ;
		}	
	}

	tw_dma_hook(	btv, RISC_SLOT_O_FIELD, &set->top->top, 0 ) ;
	tw_dma_hook(	btv, RISC_SLOT_E_FIELD, &set->bottom->bottom, 1 ) ;

}




int
tw_buffer_activate_video
(
	struct bttv *btv,
	struct bttv_buffer_set *set
)
{

	DBG_SWITCH( "----->tw_buffer_activate_video-%d\n", btv->c.nr ) ;

	/* video capture */
	if (NULL != set->top  &&  NULL != set->bottom) {

		DBG_SWITCH( "Top: %x / Bottom: %x \n", set->top, set->bottom ) ;

		
		if (set->top == set->bottom) {
		
			set->top->vb.state = STATE_ACTIVE;
			
			if (set->top->vb.queue.next)
				list_del(&set->top->vb.queue);
				
		} else {
		
			set->top->vb.state    = STATE_ACTIVE;
			
			set->bottom->vb.state = STATE_ACTIVE;
			
			if (set->top->vb.queue.next)
				list_del(&set->top->vb.queue);
			
			if (set->bottom->vb.queue.next)
				list_del(&set->bottom->vb.queue);
	
		}
	
		bttv_apply_geo(btv, &set->top->geo, 1);

		// ST_0126		
		// bttv_apply_geo(btv, &set->bottom->geo,0);
		bttv_apply_geo(btv, &set->bottom->geo, 1 ) ;
		
		// ST_0407 
		// Top will be equal bottom because it is Interlace mode ...
		// in tw_irq_next_video() ...

		tw_dma_hook(	btv, RISC_SLOT_O_FIELD, &set->top->top, 0 /*set->top_irq*/ ) ;

		tw_dma_hook(	btv, RISC_SLOT_E_FIELD, &set->bottom->bottom, 1 /*set->frame_irq*/ ) ;


		btaor( 	(set->top->btformat & TW6801_VDMAC_COLORF) | 
				(set->bottom->btformat & TW6801_VDMAC_COLORF ),
		      		~(TW6801_VDMAC_COLORF), TW6801_VDMAC ) ;
		
		btaor(	(set->top->btswap & TW6801_VDMAC_BSWAP) | 
				(set->bottom->btswap & TW6801_VDMAC_BSWAP ),
		      		~(TW6801_VDMAC_BSWAP), TW6801_VDMAC);
		
	} else if (NULL != set->top) {

		if( btv->c.nr == DBG_DeviceNumber  )
			DBG_MMAP( "TOP is not NULL !!!\n" ) ;
	
		set->top->vb.state  = STATE_ACTIVE;
		
		if (set->top->vb.queue.next)
			list_del(&set->top->vb.queue);

		bttv_apply_geo(btv, &set->top->geo,1);
		// bttv_apply_geo(btv, &set->top->geo,0);

		// ST_0126	
		//tw_dma_hook( btv, RISC_SLOT_O_FIELD, &set->top->top, 1 ) ;
		//tw_dma_hook( btv, RISC_SLOT_E_FIELD, NULL, 0 ) ;
		tw_dma_hook( btv, RISC_SLOT_O_FIELD, &set->top->top, set->frame_irq ) ;
		tw_dma_hook( btv, RISC_SLOT_E_FIELD, NULL, 0 ) ;

		btaor(	set->top->btformat & TW6801_VDMAC_COLORF, 
				~(TW6801_VDMAC_COLORF), TW6801_VDMAC );
		
		btaor(	set->top->btswap & TW6801_VDMAC_BSWAP,  
				~(TW6801_VDMAC_BSWAP), TW6801_VDMAC );

	} else if (NULL != set->bottom) {

		if( btv->c.nr == DBG_DeviceNumber )
			DBG_MMAP( "BOTTOM is not NULL !!!\n" ) ;

		set->bottom->vb.state = STATE_ACTIVE;

		if (set->bottom->vb.queue.next)
			list_del(&set->bottom->vb.queue);

		// ST_TEST_1003
		// bttv_apply_geo(btv, &set->bottom->geo,1);
		bttv_apply_geo(btv, &set->bottom->geo,0);

		// ST_0126
		// tw_dma_hook( btv, RISC_SLOT_O_FIELD, NULL, 0 ) ;		
		//tw_dma_hook( btv, RISC_SLOT_E_FIELD, &set->bottom->bottom, 1 ) ;
		tw_dma_hook( btv, RISC_SLOT_O_FIELD, NULL, 0 ) ;		
		tw_dma_hook( btv, RISC_SLOT_E_FIELD, &set->bottom->bottom, set->frame_irq) ;


		btaor(	set->top->btformat & TW6801_VDMAC_COLORF, 
				~(TW6801_VDMAC_COLORF), TW6801_VDMAC );
		
		btaor(	set->top->btswap & TW6801_VDMAC_BSWAP,  
				~(TW6801_VDMAC_BSWAP), TW6801_VDMAC );

	} else {

		if( btv->c.nr == DBG_DeviceNumber  )	
			DBG_MMAP( " ===== TOP=BOTTOM=> NULL , Set tw_dma_hook to NULL ===== \n" ) ;

		// ST_0409
		// tw_dma_hook(btv, RISC_SLOT_O_FIELD, NULL, 0);
		// tw_dma_hook(btv, RISC_SLOT_E_FIELD, NULL, 0);

		tw_dma_hook(btv, RISC_SLOT_O_FIELD, NULL, 1);
		tw_dma_hook(btv, RISC_SLOT_E_FIELD, NULL, 1);


		
	}

	DBG_SWITCH( "<-----tw_buffer_activate_video - %d\n",  btv->c.nr) ;
		
	return 0;
	
}



/* ---------------------------------------------------------- */

/* calculate geometry, build risc code */
int
tw_buffer_risc
(
	struct bttv *btv, 
	struct bttv_buffer *buf
)
{
	const struct bttv_tvnorm *tvnorm = bttv_tvnorms + buf->tvnorm;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
	struct videobuf_dmabuf *dma=videobuf_to_dma(&buf->vb);
#endif

	// ST_0116
	DBG_SWITCH( "----->tw_buffer_risc - %d:: bf: %s  buffer: %x  depth: %d size: %d * %d flag:%x \n",
		btv->c.nr, v4l2_field_names[buf->vb.field], dma, buf->fmt->depth, 
		buf->vb.width, buf->vb.height, buf->fmt->flags );

	/* pixel modes */
	if (buf->fmt->flags & FORMAT_FLAGS_PACKED) {

		int bpl = (buf->fmt->depth >> 3) * buf->vb.width;
		
		int bpf = bpl * (buf->vb.height >> 1);

		DBG_SWITCH( "--->tw_buffer_risc: bpl=%d, bpf=%d\n", bpl, bpf ) ;
		
		bttv_calc_geo( 	btv,
					  	&buf->geo,
					  	buf->vb.width,
					  	buf->vb.height,
			      			V4L2_FIELD_HAS_BOTH(buf->vb.field),
			      			buf->tvnorm ) ;

		switch (buf->vb.field) {
		case V4L2_FIELD_TOP:
			
			DBG_SWITCH( "V4L2_FIELD_TOP" ) ;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
			tw_dma_packed(	btv,
							&buf->top,
							dma->sglist,
					 		0,				// Offset
					 		bpl,				// bpl
					 		0,				// Padding
					 		buf->vb.height ) ;	// Lines

#else

			tw_dma_packed(	btv,
							&buf->top,
							buf->vb.dma.sglist,
					 		0,				// Offset
					 		bpl,				// bpl
					 		0,				// Padding
					 		buf->vb.height ) ;	// Lines
#endif

			break;
			
		case V4L2_FIELD_BOTTOM:
			
			DBG_SWITCH( "V4L2_FIELD_BOTTOM" ) ;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
			tw_dma_packed(	btv,
							&buf->bottom,
							dma->sglist,
					 		0,				// Offset
					 		bpl,				// bpl
					 		0,				// Padding
					 		buf->vb.height ) ;	// Lines

#else
			tw_dma_packed(	btv,
							&buf->bottom,
							buf->vb.dma.sglist,
					 		0,				// Offset
					 		bpl,				// bpl
					 		0,				// Padding
					 		buf->vb.height ) ;	// Lines
#endif

			break;

		case V4L2_FIELD_INTERLACED:

			DBG( "dev: %d --> V4L2_FIELD_INTERLACED( sglist: %x )\n", btv->c.nr,  dma->sglist) ;

	// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
 			tw_dma_packed(	btv,
							&buf->top,
							dma->sglist,
					 		0,
					 		bpl,
					 		bpl,
					 		buf->vb.height >> 1 ) ;

			tw_dma_packed(	btv,
							&buf->bottom,
							dma->sglist,
					 		bpl, 
					 		bpl,
					 		bpl,
					 		buf->vb.height>>1 ) ;


			DBG("---->TOP: %8x,  BOTTOM: %8x \n", buf->top, buf->bottom ) ;


#else
 			tw_dma_packed(	btv,
							&buf->top,
							buf->vb.dma.sglist,
					 		0,
					 		bpl,
					 		bpl,
					 		buf->vb.height >> 1 ) ;

			DBG("---->BOTTOM ( %x )... \n", buf->bottom ) ;	
			tw_dma_packed(	btv,
							&buf->bottom,
							buf->vb.dma.sglist,
					 		bpl, 
					 		bpl,
					 		bpl,
					 		buf->vb.height>>1 ) ;
#endif

			break;
			
		case V4L2_FIELD_SEQ_TB:
			
			DBG_SWITCH( "V4L2_FIELD_SEQ_TB-%d ", btv->c.nr ) ;

// ST_0115, for kernel 2.6.27 support 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,17)	
			tw_dma_packed(	btv,
							&buf->top,
							dma->sglist,
					 		0,
					 		bpl,
					 		0,
					 		buf->vb.height >> 1);

			tw_dma_packed(	btv,
							&buf->bottom,
							dma->sglist,
					 		bpf,
					 		bpl,
					 		0,
					 		buf->vb.height >> 1);

#else
			tw_dma_packed(btv,&buf->top,buf->vb.dma.sglist,
					 0,bpl,0,buf->vb.height >> 1);
			tw_dma_packed(btv,&buf->bottom,buf->vb.dma.sglist,
					 bpf,bpl,0,buf->vb.height >> 1);
#endif
			break;
		default:
			BUG();
		}
	}

	/* copy format info */
	buf->btformat = buf->fmt->btformat;
	buf->btswap   = buf->fmt->btswap;

	DBG_SWITCH( "<-----tw_buffer_risc\n" ) ;
	
	return 0;

}

/* ---------------------------------------------------------- */

/* calculate geometry, build risc code */
int
bttv_overlay_risc(struct bttv *btv,
		  struct bttv_overlay *ov,
		  const struct bttv_format *fmt,
		  struct bttv_buffer *buf)
{

	/* check interleave, bottom+top fields */

	dprintk(KERN_DEBUG
		"bttv%d: overlay fields: %s format: %s  size: %dx%d\n",
		btv->c.nr, v4l2_field_names[buf->vb.field],
		fmt->name,ov->w.width,ov->w.height);

	/* calculate geometry */
	bttv_calc_geo(btv,&buf->geo,ov->w.width,ov->w.height,
		      V4L2_FIELD_HAS_BOTH(ov->field), ov->tvnorm);

	/* build risc code */
	switch (ov->field) {
	case V4L2_FIELD_TOP:
		bttv_risc_overlay(btv, &buf->top,    fmt, ov, 0, 0);
		break;
	case V4L2_FIELD_BOTTOM:
		bttv_risc_overlay(btv, &buf->bottom, fmt, ov, 0, 0);
		break;
	case V4L2_FIELD_INTERLACED:
		bttv_risc_overlay(btv, &buf->top,    fmt, ov, 0, 1);
		bttv_risc_overlay(btv, &buf->bottom, fmt, ov, 1, 0);
		break;
	default:
		BUG();
	}

	/* copy format info */
	buf->btformat = fmt->btformat;
	buf->btswap   = fmt->btswap;
	buf->vb.field = ov->field;
	return 0;
}


// ST_0508_ADD for scale value setting ...
# if 0
void
tw_set_scale
(
	struct bttv *btv
)
}


	return ;
{

#endif



/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
